#include "General.h"
#include "CustomFlipKill.h"
#include "VehicleGameObjDef.h"
#include "HashTemplateClass.h"
#include "HashTemplateIterator.h"
#include "slist.h"
#include "VehicleGameObj.h"
#include "GameObjManager.h"
#include "engine_tt.h"
#include "engine_io.h"
#include "gmgame.h"

HashTemplateClass<StringClass, int> DontFlipKillPresets; // Contains vehicle presets that won't get flip killed
HashTemplateClass<StringClass, int> RotateUpwardsPresets; // These presets get rotated upwards if they flip
int LastExecuteTime = 0; // Used by OnThink() stores the last time OnThink() was fully called
int CheckInterval = 3; // Used by OnThink
float ScriptCheckingInterval = 1.f; // Used by CustomFlipKill_Script::Create()
int ChecksCount = 4; // Amount of times to check between ScriptCheckingInterval before destroying the vehicle
					// ^ Used by CustomFlipKill_Script::Timer_Expired()

CustomFlipKill::CustomFlipKill()
{
	RegisterEvent(EVENT_GLOBAL_INI,this);
	RegisterEvent(EVENT_THINK_HOOK,this);
	RegisterEvent(EVENT_LOAD_LEVEL_HOOK,this);
}

CustomFlipKill::~CustomFlipKill()
{
	UnregisterEvent(EVENT_GLOBAL_INI,this);
	UnregisterEvent(EVENT_THINK_HOOK,this);
	UnregisterEvent(EVENT_LOAD_LEVEL_HOOK,this);
}
void CustomFlipKill::OnLoadLevel()
{
	LastExecuteTime = 0;
}
		
void CustomFlipKill::OnLoadGlobalINISettings(INIClass *SSGMIni)
{
	CheckInterval = SSGMIni->Get_Int("CustomFlipKill", "CheckInterval", 2);
	ChecksCount = SSGMIni->Get_Int("CustomFlipKill", "ChecksCount", 4);
	ScriptCheckingInterval =  SSGMIni->Get_Float("CustomFlipKill", "ScriptCheckingInterval", 1.f);

	int Count = SSGMIni->Entry_Count("CustomFlipKill_DontFlipKillPresets");
	for (int i = 0; i < Count; i++)
	{
		const char *Entry = SSGMIni->Get_Entry("CustomFlipKill_DontFlipKillPresets", i);
		StringClass Insert;
		SSGMIni->Get_String(Insert, "CustomFlipKill_DontFlipKillPresets", Entry);
		DontFlipKillPresets.Insert(Insert, 1);
	}

	Count = SSGMIni->Entry_Count("CustomFlipKill_RotateUpwardsPresets");
	for (int i = 0; i < Count; i++)
	{
		const char *Entry = SSGMIni->Get_Entry("CustomFlipKill_RotateUpwardsPresets", i);
		StringClass Insert;
		SSGMIni->Get_String(Insert, "CustomFlipKill_RotateUpwardsPresets", Entry);
		RotateUpwardsPresets.Insert(Insert, 1);
	}
}

void CustomFlipKill::OnThink()
{
	if (!The_Game())
	{
		return;
	}
	
	int CurrentExecuteTime = The_Game()->Get_Game_Duration_S();
	if (CurrentExecuteTime < (LastExecuteTime + CheckInterval)) return;

	LastExecuteTime = CurrentExecuteTime;
//	Console_Output("LastExecuteTime updated to %d\n", LastExecuteTime); // DEBUG CRAP
	
	SLNode<SmartGameObj> *x = GameObjManager::SmartGameObjList.Head();
	while (x)
	{
		GameObject *o = As_VehicleGameObj(x->Data());
		if (o)
		{
			if (Get_Vehicle_Mode(o) != VEHICLE_TYPE_TURRET)
			{
				Matrix3D Transform = Get_Transform(o);
				float AbsRotationY = fabs(Transform.getRotationY());

				if (AbsRotationY > 1.3f)
				{
					if (RotateUpwardsPresets.Get(Commands->Get_Preset_Name(o), 0))
					{
						if (!Is_Script_Attached(o, "RotateUpwards_Script"))
						{
							Attach_Script_Once(o, "RotateUpwards_Script", "");
						}
					}
					else if (DontFlipKillPresets.Get(Commands->Get_Preset_Name(o), 0) == 0)
					{
						if (!Is_Script_Attached(o, "CustomFlipKill_Script"))
						{
							Attach_Script_Once(o, "CustomFlipKill_Script", "");
						}
					}
				}
			}
		}
		x = x->Next();
	}
}

void CustomFlipKill_Script::Created(GameObject *obj)
{
	Commands->Start_Timer(obj, this, ScriptCheckingInterval, 1);
}

void CustomFlipKill_Script::Timer_Expired(GameObject *obj, int number)
{
	if (number == 1)
	{
		Matrix3D Transform = Get_Transform(obj);
		float AbsRotationY = fabs(Transform.getRotationY());

		if ((AbsRotationY > 1.3f)) // If the vehicle is flipped
		{
			TimesChecked++;

			if (TimesChecked >= ChecksCount)
			{
				// Destroy vehicle
				Commands->Apply_Damage(obj, 9999.f, "Death", 0);
				return;
			}
			Commands->Start_Timer(obj, this, ScriptCheckingInterval, 1);
		}
		else // During one of our checks the vehicle "unflipped" itself so we can destroy this script
		{
			Destroy_Script();
		}
	}
}

ScriptRegistrant<CustomFlipKill_Script> CustomFlipKill_Script_Registrant("CustomFlipKill_Script","");

void RotateUpwards_Script::Created(GameObject *obj)
{
	Commands->Start_Timer(obj, this, 0.5f, 1);
}

void RotateUpwards_Script::Timer_Expired(GameObject *obj, int number)
{
	if (number == 1)
	{
		Matrix3D Transform = Get_Transform(obj);
		float AbsRotationY = fabs(Transform.getRotationY());

		if ((AbsRotationY > 1.3f)) // If the vehicle is flipped
		{
			// Rotate our vehicle upwards
			Matrix3D Transform = Get_Transform(obj);
			Transform.Rotate_X(DEG_TO_RADF(25));
			Set_Transform(obj,Transform);

			// We need to higher the position of our vehicle slightly
			Vector3 Pos = Commands->Get_Position(obj);
			Pos.Z += 3.0f;
			Commands->Set_Position(obj, Pos);

			Commands->Start_Timer(obj, this, 0.5f, 1);
		}
		else // During one of our checks the vehicle "unflipped" itself so we can destroy this script
		{
			Destroy_Script();
		}
	}
}

ScriptRegistrant<RotateUpwards_Script> RotateUpwards_Script_Registrant("RotateUpwards_Script","");

CustomFlipKill customFlipKill;

extern "C" __declspec(dllexport) Plugin* Plugin_Init()
{
	return &customFlipKill;
}
